/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//////////////////////////////////////////////////////////////////////////
// Thread.h
//
// This file contains an OS independent interface for thread manipulation
// A Thread class is defined for easy usage.
//
// Usage:
//
// Option 1: Construct an instance of the "Thread" class with an external
// callback function. When calling the "start" method, the thread will be
// started on the callback function
//
// Option 2: Subclass the "Thread" class and reimplement the virtual "run"
// method. When calling the "start" method, the thread will be started
// on the "run" function.
//
// Implementation overview:
// Calling the "start" method will start the new thread, which will call the
// "run" method. The default implementation of the "run" method will call
// the Callback function given in the constructor.
//
//////////////////////////////////////////////////////////////////////////
#ifndef _LAD_THREAD_H
#define _LAD_THREAD_H

#ifndef WAIT_INFINITE
#define WAIT_INFINITE 0xffffffff
#endif
#ifndef NULL
#define NULL 0
#endif

typedef void (*CallbackFunction) (void*);

class OSThread;

class Thread {
	friend class OSThread;
public:
	Thread (CallbackFunction func = NULL, void* param = NULL);
 	Thread (const Thread& rhs);
	virtual ~Thread();

    // wait for the thread to complete; return true if the thread completed,
	// false on timeout
	bool wait(unsigned long msecs = WAIT_INFINITE) const;
	// start the new thread, return true on success
    bool start();
	// true if the thread is in running state
    bool running() const;
	// measure the time (in msecs) from thread start-time
	long elapsedTime() const;

	// return ID for the current thread
	static unsigned long currentThread();
	// put the current thread to sleep
	static void msleep (long msecs);

	Thread& operator= (const Thread& rhs);
protected:
	virtual void run();
private:
	CallbackFunction _func;
	void* _param;
	long _startTime;
	OSThread* _osThread;
};

#endif //_LAD_THREAD

